import flet as ft
import json
import os
import requests

SAVE_FILE = "saved_messages.json"

def main(page: ft.Page):
    page.title = "PyZen Webhook Builder ~ the_femboy.dev"
    page.theme_mode = ft.ThemeMode.DARK
    page.scroll = "auto"
    page.window_width = 1200
    page.window_height = 800
    page.window_resizable = True

    # Controls
    webhook_url = ft.TextField(label="Webhook URL", expand=True)
    message_content = ft.TextField(label="Message Content", multiline=True, expand=True)

    embed_title = ft.TextField(label="Embed Title", expand=True)
    embed_description = ft.TextField(label="Embed Description", multiline=True, expand=True)
    embed_footer = ft.TextField(label="Footer Text", expand=True)
    embed_footer_icon = ft.TextField(label="Footer Icon URL", expand=True)
    embed_image = ft.TextField(label="Image URL", expand=True)
    embed_thumbnail = ft.TextField(label="Thumbnail URL", expand=True)

    field_name = ft.TextField(label="Field Name", expand=True)
    field_value = ft.TextField(label="Field Value", expand=True)
    inline_checkbox = ft.Checkbox(label="Inline Field", value=False)

    embed_color = ft.TextField(label="Embed Color (#hex)", expand=False, value="#2f3136")
    color_display = ft.Container(width=30, height=30, bgcolor=embed_color.value, border_radius=5)

    send_button = ft.ElevatedButton("Send Webhook", on_click=lambda e: send_webhook(e))

    embed_fields = []
    embed_buttons = []
    saved_messages = {}

    fields_column = ft.Column()
    buttons_column = ft.Column()
    saved_dropdown = ft.Dropdown(label="Saved Messages", options=[], on_change=lambda e: load_selected(e.control.value))

    # Live Preview Controls
    preview_embed_title = ft.Text(value="", size=18, weight="bold")
    preview_embed_description = ft.Text(value="", size=14)
    preview_embed_fields = ft.Column()
    preview_embed_footer = ft.Row()
    preview_embed_image = ft.Image(src="", width=300, visible=False)
    preview_embed_thumbnail = ft.Image(src="", width=50, height=50, visible=False)

    preview_embed = ft.Container(
        content=ft.Stack([
            ft.Container(
                content=ft.Column([
                    ft.Row([
                        ft.Column([
                            preview_embed_title,
                            preview_embed_description,
                            preview_embed_fields,
                            preview_embed_image,
                            preview_embed_footer
                        ], expand=True),
                        ft.Container(content=preview_embed_thumbnail, alignment=ft.alignment.top_right, margin=10),
                    ])
                ]),
                bgcolor=ft.Colors.BLUE_GREY_900,
                padding=15,
                border_radius=10,
                border=ft.border.all(1, ft.Colors.BLUE_GREY_700),
                width=600
            )
        ]),
        width=600
    )

    def update_preview(_=None):
        preview_embed_title.value = embed_title.value
        preview_embed_description.value = embed_description.value

        # Fields
        preview_embed_fields.controls.clear()
        inline_row = ft.Row(wrap=True, spacing=10)
        for field in embed_fields:
            field_box = ft.Container(
                content=ft.Column([
                    ft.Text(field["name"], weight="bold", size=12),
                    ft.Text(field["value"], size=11),
                ]),
                bgcolor=ft.Colors.with_opacity(0.05, ft.Colors.WHITE),
                padding=6,
                border_radius=4,
                width=150 if field["inline"] else None
            )
            if field["inline"]:
                inline_row.controls.append(field_box)
            else:
                preview_embed_fields.controls.append(field_box)
        
        if inline_row.controls:
            preview_embed_fields.controls.insert(0, inline_row)

        # Footer
        preview_embed_footer.controls.clear()
        if embed_footer.value:
            if embed_footer_icon.value.strip():
                preview_embed_footer.controls.append(
                    ft.Image(src=embed_footer_icon.value, width=18, height=18)
                )
            preview_embed_footer.controls.append(
                ft.Text(embed_footer.value, size=10, italic=True)
            )

        # Images
        preview_embed_image.visible = bool(embed_image.value.strip())
        preview_embed_image.src = embed_image.value.strip()

        preview_embed_thumbnail.visible = bool(embed_thumbnail.value.strip())
        preview_embed_thumbnail.src = embed_thumbnail.value.strip()

        # Color box
        color_display.bgcolor = embed_color.value

        page.update()

    def add_field(e):
        if field_name.value and field_value.value:
            field = {
                "name": field_name.value,
                "value": field_value.value,
                "inline": inline_checkbox.value
            }
            embed_fields.append(field)
            fields_column.controls.append(
                ft.Text(f"{field['name']}: {field['value']} (Inline: {field['inline']})")
            )
            field_name.value = field_value.value = ""
            inline_checkbox.value = False
            update_preview()

    def remove_field(e):
        if embed_fields:
            embed_fields.pop()
            fields_column.controls.pop()
            update_preview()

    def remove_button(e):
        if embed_buttons:
            embed_buttons.pop()
            buttons_column.controls.pop()
            page.update()

    def save_message(e):
        if not webhook_url.value.strip():
            return

        message_data = {
            "webhook_url": webhook_url.value,
            "content": message_content.value,
            "embed": {
                "title": embed_title.value,
                "description": embed_description.value,
                "footer": {
                    "text": embed_footer.value,
                    "icon_url": embed_footer_icon.value,
                },
                "image": {"url": embed_image.value},
                "thumbnail": {"url": embed_thumbnail.value},
                "color": embed_color.value,
                "fields": embed_fields
            },
            "buttons": embed_buttons
        }

        name = embed_title.value or "Untitled"
        saved_messages[name] = message_data

        with open(SAVE_FILE, "w") as f:
            json.dump(saved_messages, f, indent=4)

        page.snack_bar = ft.SnackBar(ft.Text("Message saved!"))
        page.snack_bar.open = True
        page.update()

    def load_selected(name):
        if name and name in saved_messages:
            load_message(saved_messages[name])

    def load_message(data=None):
        if not data:
            if os.path.exists(SAVE_FILE):
                with open(SAVE_FILE, "r") as f:
                    loaded = json.load(f)
                global saved_messages
                saved_messages = loaded
            return
        
        webhook_url.value = data.get("webhook_url", "")
        message_content.value = data.get("content", "")
        embed = data.get("embed", {})
        embed_title.value = embed.get("title", "")
        embed_description.value = embed.get("description", "")
        embed_footer.value = embed.get("footer", {}).get("text", "")
        embed_footer_icon.value = embed.get("footer", {}).get("icon_url", "")
        embed_image.value = embed.get("image", {}).get("url", "")
        embed_thumbnail.value = embed.get("thumbnail", {}).get("url", "")
        embed_color.value = embed.get("color", "#ff0000")

        embed_fields.clear()
        fields_column.controls.clear()
        for field in embed.get("fields", []):
            embed_fields.append(field)
            fields_column.controls.append(
                ft.Text(f"{field['name']}: {field['value']} (Inline: {field['inline']})")
            )

        update_preview()

    def send_webhook(e):
        embed = {
            "title": embed_title.value,
            "description": embed_description.value,
            "fields": [
                {
                    "name": f["name"],
                    "value": f["value"],
                    "inline": f["inline"]
                } for f in embed_fields
            ],
            "footer": {
                "text": embed_footer.value,
                "icon_url": embed_footer_icon.value
            },
            "image": {
                "url": embed_image.value
            },
            "thumbnail": {
                "url": embed_thumbnail.value
            },
            "color": int(embed_color.value.lstrip('#'), 16) if embed_color.value else 3092790
        }

        payload = {
            "content": message_content.value,
            "embeds": [embed],
            "components": []
        }

        if embed_buttons:
            buttons = []
            for b in embed_buttons:
                buttons.append({
                    "type": 2,
                    "style": 5,
                    "label": b["label"],
                    "url": b["url"]
                })
            payload["components"].append({
                "type": 1,
                "components": buttons
            })

        try:
            res = requests.post(webhook_url.value, json=payload)
            if res.status_code in (200, 204):
                page.snack_bar = ft.SnackBar(ft.Text("Webhook sent successfully!"))
            else:
                page.snack_bar = ft.SnackBar(ft.Text(f"Failed: {res.status_code}"))
        except Exception as ex:
            page.snack_bar = ft.SnackBar(ft.Text(f"Error: {ex}"))

        page.snack_bar.open = True
        page.update()

    for field in [embed_title, embed_description, embed_footer, embed_footer_icon, embed_image, embed_thumbnail, embed_color]:
        field.on_change = update_preview

    embed_color.on_change = update_preview

    load_message()

    page.add(
        ft.Row([
            ft.Container(
                content=ft.Column([
                    ft.Text("PyZen Webhook Builder", size=28, weight="bold"),
                    webhook_url,
                    message_content,
                    ft.Divider(),
                    ft.Text("Embed", size=22, weight="bold"),
                    embed_title,
                    embed_description,
                    ft.Row([embed_footer, embed_footer_icon]),
                    embed_image,
                    embed_thumbnail,
                    ft.Row([embed_color, color_display]),
                    ft.Text("Fields", size=18),
                    ft.Row([field_name, field_value, inline_checkbox]),
                    ft.Row([
                        ft.ElevatedButton("Add Field", on_click=add_field),
                        ft.ElevatedButton("Remove Field", on_click=remove_field)
                    ]),
                    buttons_column,
                    ft.Divider(),
                    send_button
                ]),
                expand=2,
                padding=10
            ),
            ft.Container(
                content=ft.Column([
                    ft.Text("Live Preview", size=24),
                    preview_embed
                ]),
                padding=10,
                expand=1
            )
        ])
    )

if __name__ == "__main__":
    ft.app(target=main)
